%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% FourierMotzkinGUI - A basic GUI for Fourier-Motzkin Elimination 
% with particular focus on applications to information theory
% Copyright (C) 2011  Joffrey Villard
%
% This program is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program.  If not, see <http://www.gnu.org/licenses/>.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function FourierMotzkinGUI
% - interface graphique pour excuter la mthode d'limination de 
% Fourier-Motzkin sur des systmes d'ingalits  paramtres "symboliques"
% comme ceux rencontrs dans les problmes de thorie de l'information
% - permet de sauvegarder/charger une session (ie, un systme) via un
% fichier .mat
% - le rsultat peut tre export dans un fichier texte

version = '1.0';
dateVer = '2011';

%====================================================================
% fentre principale
fenetre = figure(	'HandleVisibility','off',...
	'NumberTitle','off',...
	'Resize','off',...
	'MenuBar','none',...
	'ToolBar','none',...
	'DockControls','off',...
	'Position',[250 150 900 700],...
	'CloseRequestFcn',@quitFMG);

%====================================================================
% menus de navigation entre les diffrentes tapes 
% + bouton de sauvegarde
menuGrp = uipanel('Parent',fenetre,...
	'Units','normalized',...
	'Position',[0 .96 1 .04]);

%----------------------------------------------------------------
% reste de la figure, pour tous les autres contenus,
% auquel on attache les donnes (par APPdata)
% 
% - les structures 'data' et 'newData' avec les champs suivants:
%	. 'vars':	liste ('cell array' de 'string') des variables 		(1,m)
%	. 'param':	liste ('cell array' de 'string') des paramtres		(1,p)
%	. 'A':		matrice des termes de gauche						(m,n)
%	. 'M':		matrice des termes de droite						(m,p)
% elles correspondent au prbl A*vars <= M*param
% - le tableau 'steps' indiquant les tapes OK						(1,nbSteps)
% - le tableau 'varsRemain' indiquant les variables conserves		(1,nbRemain)
% - la struct 'numStep' des numro des tapes
% - le chemin complet de la session 'nomSession'

panel = uipanel('Parent',fenetre,...
	'BackgroundColor',[.8 .8 .8],...
	'Units','normalized',...
	'Position',[0 0 1 .96]);
nbSteps = 6;

numSteps = struct(...
	'entreeVars',	1,...
	'entreeParam',	2,...
	'inegalites',	3,...
	'choixVars',	4,...
	'runFME',		5,...
	'resultat',		6);
setappdata(panel,'numSteps',numSteps);


%====================================================================
% ensemble des menus ('Save' + 'Start' + steps)
offset = 2;
menus = zeros(1,nbSteps+offset);

% sauvegarde du systme  tout moment
menus(1) = uicontrol(menuGrp,'Style','pushbutton',...
	'Tag','menus',...
	'String','SAVE session',...
	'Enable','off',...
	'FontSize',8,...
    'Units','normalized',...
	'Position',[0 0 .100 1],...
	'Callback',@saveSession);

% retour au dbut
uicontrol(menuGrp,'Style','text','String','|','FontSize',13,...
    'Units','normalized','Position',[.100 0 .025 1]);
menus(2) = uicontrol(menuGrp,'Style','pushbutton',...
	'Tag','menus',...
	'String','START',...
	'FontSize',8,...
    'Units','normalized',...
	'Position',[.125 0 .100 1],...
	'Callback',@init);

%--------------------------------------------------------------------
% les tapes

% entre des variables
uicontrol(menuGrp,'Style','text','String','>>','FontSize',12,...
    'Units','normalized','Position',[.225 0 .025 .9]);
menus(numSteps.entreeVars+offset) = uicontrol(menuGrp,'Style','pushbutton',...
	'Tag','menus',...
	'String','Enter variables',...
	'FontSize',8,...
    'Units','normalized',...
	'Position',[.250 0 .100 1],...
	'Callback',{@FMGentreeVarsParam,panel,numSteps.entreeVars});

% entre des paramtres
uicontrol(menuGrp,'Style','text','String','>>','FontSize',12,...
    'Units','normalized','Position',[.350 0 .025 .9]);
menus(numSteps.entreeParam+offset) = uicontrol(menuGrp,'Style','pushbutton',...
	'Tag','menus',...
	'String','Enter parameters',...
	'FontSize',8,...
    'Units','normalized',...
	'Position',[.375 0 .110 1],...
	'Callback',{@FMGentreeVarsParam,panel,numSteps.entreeParam});

% entre des ingalits
uicontrol(menuGrp,'Style','text','String','>>','FontSize',12,...
    'Units','normalized','Position',[.485 0 .025 .9]);
menus(numSteps.inegalites+offset) = uicontrol(menuGrp,'Style','pushbutton',...
	'Tag','menus',...
	'String','Enter inequalities',...
	'FontSize',8,...
    'Units','normalized',...
	'Position',[.510 0 .110 1],...
	'Callback',{@FMGinegalites,panel});

% choix des variables restantes
uicontrol(menuGrp,'Style','text','String','>>','FontSize',12,...
    'Units','normalized','Position',[.620 0 .025 .9]);
menus(numSteps.choixVars+offset) = uicontrol(menuGrp,'Style','pushbutton',...
	'Tag','menus',...
	'String','Choose remaining variables',...
	'FontSize',8,...
    'Units','normalized',...
	'Position',[.645 0 .160 1],...
	'Callback',{@FMGchoixVars,panel});

% Fourier-Motzkin proprement dit (seulement du texte)
uicontrol(menuGrp,'Style','text','String','>>','FontSize',12,...
    'Units','normalized','Position',[.805 0 .025 .9]);
menus(numSteps.runFME+offset) = uicontrol(menuGrp,'Style','text',...
	'Tag','menus',...
	'String','Run FME',...
	'FontSize',8,...
    'Units','normalized',...
	'Position',[.830 0 .060 .75]);

% affichage du rsultat
uicontrol(menuGrp,'Style','text','String','>>','FontSize',12,...
    'Units','normalized','Position',[.890 0 .025 .9]);
menus(numSteps.resultat+offset) = uicontrol(menuGrp,'Style','pushbutton',...
	'Tag','menus',...
	'String','See the result',...
	'FontSize',8,...
    'Units','normalized',...
	'Position',[.915 0 .085 1],...
	'Callback',{@FMGresultat,panel});

%----------------------------------------------------------------
% on conserve un lien vers ces menus
setappdata(panel,'menus',menus);

init([],[]);


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% cran initial: 2 boutons (charger ou crer un systme)
	function init(ho,ev)
		% SI l'appel  cette fonction fait suite  l'appui sur un bouton
		if ~isempty(ho)
			reponse = strcmp(questdlg('Clear current session and restart?','Warning','Yes','No','No'),'Yes');
		else
			reponse = 1;
		end
		
		if reponse
			% on supprime les lments relatifs  d'autres tapes
			delete(get(panel,'Children'));

			% titre de la fentre
			setSessionName('');

			% gestion des menus	(tous 'off')
			set(findall(menuGrp,'Tag','menus'),'Enable','off');

			% version + date + copyright
			uicontrol(panel,'Style','text',...
				'String',sprintf(' FourierMotzkinGUI %s %s%s',version,char(169),dateVer),...
				'FontSize',14,...
				'ForegroundColor',[.6 .6 .6],...
				'BackgroundColor',[.8 .8 .8],...
				'HorizontalAlignment','left',...
				'Units','normalized',...
				'Position',[0 0 .5 .04]);
			
			% about
			uicontrol(panel,'Style','pushbutton',...
				'String','About',...
				'FontSize',14,...
				'ForegroundColor',[.6 .6 .6],...
				'Units','normalized',...
				'Position',[.7 0 .15 .05],...
				'Callback',{@FMGabout,panel});
			
			% userguide
			uicontrol(panel,'Style','pushbutton',...
				'String','Userguide',...
				'FontSize',14,...
				'ForegroundColor',[.6 .6 .6],...
				'Units','normalized',...
				'Position',[.85 0 .15 .05],...
				'Callback','open(''userguide.pdf'')');
							
			
			% boutons 'Load...' et 'Create...'
			uicontrol(  panel,'Style','pushbutton',...
				'String','Load an existing session',...
				'FontSize',15,...
				'Units','normalized',...
				'Position',[.2 .6 .6 .2],...
				'Callback',@openSession);

			uicontrol(  panel,'Style','pushbutton',...
				'String','Create a new one',...
				'FontSize',15,...
				'Units','normalized',...
				'Position',[.2 .2 .6 .2],...
				'Callback',@newSession);
		end
	end

		
%----------------------------------------------------------------
% ouverture d'un systme (ie, la structure 'data') enregistr dans un fichier '*.mat'
	function openSession(ho,ev)
		[filename, pathname] = uigetfile('*.mat', 'Pick a MAT-file');
		if ~isequal(filename,0)
			try
				% rcupration des donnes
				load(fullfile(pathname,filename),'data','newData','steps','varsRemain');
				% vrification de la cohrence des dimensions:
				% - nbre de variables = nbre de colonnes de A
				% - nbre de paramtres = nbre de colonnes de M
				% - autant de lignes dans A et M
				% nbre total d'tapes OK
				loadOK =	(length(data.vars)	== size(data.A,2))...
						&&	(length(data.param)	== size(data.M,2))...
						&&	(size(data.A,1)		== size(data.M,1))...
						&&	(length(newData.vars)	== size(newData.A,2))...
						&&	(length(newData.param)	== size(newData.M,2))...
						&&	(size(newData.A,1)		== size(newData.M,1))...
						&&	(length(steps) == nbSteps);
			catch
				loadOK = 0;
			end
			
			% - - - - - - - - - - - - - - - - - - - - - - - - - - - -
			% SI le chargement est russi, 
			% on passe  la dernire tape enregistre			
			if loadOK								
				% on attache les donnes  la fentre
				setappdata(panel,'data',data);
				setappdata(panel,'newData',newData);
				setappdata(panel,'steps',steps);
				setappdata(panel,'varsRemain',varsRemain);
				
				% nom de la session
				setSessionName(fullfile(pathname,filename));
				
				% menus
				FMGmajMenus([],[],panel,0);	
		
				% on regarde la dernire tape non OK 
				% pour repartir juste avant
				nOK = find(steps==0,1);
				if isempty(nOK)
					lastStep = nbSteps;
				else
					lastStep = nOK-1;
				end
				switch lastStep
					case numSteps.entreeVars
						% entre des variables
						FMGentreeVarsParam([],[],panel,numSteps.entreeVars);
						
					case numSteps.entreeParam
						% entre des paramtres
						FMGentreeVarsParam([],[],panel,numSteps.entreeParam);
						
					case numSteps.inegalites
						% entre des ingalits
						FMGinegalites([],[],panel);
						
					case {numSteps.choixVars, numSteps.runFME}
						% choix des variables restantes
						FMGchoixVars([],[],panel);						
						
					case numSteps.resultat
						% affichage du rsultat
						FMGresultat([],[],panel);
						
					otherwise
						loadOK = 0;
				end
			end
				
			% - - - - - - - - - - - - - - - - - - - - - - - - - - - -
			% SINON, erreur !
			if ~loadOK				
				uiwait(msgbox('This file can not be loaded in FourierMotzkinGUI.','Error','error','modal'));
			end
		end
	end

%----------------------------------------------------------------
% cration d'un systme 
% (ie, initialisation de 'data' et entre des variables et paramtres)
	function newSession(ho,ev)
		delete(get(panel,'Children'));
				
		% donnes su systme initial
		data = struct('vars',{{}},'param',{{}},'A',{[]},'M',{[]});
		
		% donnes du systme rduit
		newData = struct('vars',{{}},'param',{{}},'A',{[]},'M',{[]});
		
		% on attache ces donnes au panel
		setappdata(panel,'data',data);
		setappdata(panel,'newData',newData);
		setappdata(panel,'steps',zeros(1,nbSteps));
		setappdata(panel,'varsRemain',[]);
		
		% nom de la session
		setSessionName('Untitled');
		
		% menus
		FMGmajMenus([],[],panel,1);	
		
		% entre des variables et paramtres
		FMGentreeVarsParam([],[],panel,numSteps.entreeVars);
	end



%----------------------------------------------------------------
% sauvegarde de la session en cours
	function saveSession(ho,ev)
		[filename,pathname,filterindex] =...
			uiputfile('*.mat','Save session as',getappdata(panel,'nomSession'));
		if (filterindex>0)
			data = getappdata(panel,'data');
			newData = getappdata(panel,'newData');
			steps = getappdata(panel,'steps');
			varsRemain = getappdata(panel,'varsRemain');
			save(fullfile(pathname,filename),'data','newData','steps','varsRemain');
			
			% nouveau nom de la session
			setSessionName(fullfile(pathname,filename));
		end
	end


%----------------------------------------------------------------
% nom de la fentre principale,  partir d'un chemin complet de fichier
	function setSessionName(str)
		commun = sprintf('FourierMotzkinGUI %s',version);
		setappdata(panel,'nomSession',str);
		if isempty(str)
			set(fenetre,'Name',commun);
		else
			[pathstr, nom] = fileparts(str);
			set(fenetre,'Name',sprintf('%s - %.50s',commun,nom));
		end		
	end


%--------------------------------------------------------------------
%  la fermeture de la fentre
	function quitFMG(ho,ev)
		% SI une session non sauvegarde est ouverte 
		% (il y a '*'  la fin du nom)
		nom = get(fenetre,'Name');
		if strcmp(nom(end),'*')
			reponse = strcmp(questdlg('Quit FMG? (unsaved data will be lost)','Warning','Yes','No','No'),'Yes');
		else
			reponse = 1;
		end
		if reponse
			delete(fenetre);
		end
	end

end